unit GLClearHistory ;

interface

uses Classes, SysUtils, Windows, Dialogs, Registry, ToolIntf, ExptIntf ;

type
    TGLClearHistoryExpert = class(TIExpert)
    private
       FClearRequested : boolean ;
       FMenuItem : TIMenuItemIntf ;
       procedure ClearHistory ;
    public
       constructor Create ;
       destructor Destroy ; override ;
       procedure Execute ; override ;
       procedure MenuClick(Sender : TIMenuItemIntf) ;
       function GetAuthor : string ; override ;
       function GetComment : string ; override ;
       function GetGlyph : HICON ; override ;
       function GetIDString : string ; override ;
       function GetMenuText : string ; override ;
       function GetName : string ; override ;
       function GetPage : string ; override ;
       function GetState : TExpertState ; override ;
       function GetStyle : TExpertStyle ; override ;
    end ;

procedure Register ;

implementation

const
    FILE_REOPEN_ID = 'FileClosedFilesItem' ;

{ TGLClearHistoryExpert }

constructor TGLClearHistoryExpert.Create ;
var
   TargetItem : TIMenuItemIntf ;
   ParentItem : TIMenuItemIntf ;
   Index : integer ;
   MenuOptions : TIMenuFlags ;
begin
     inherited Create ;
     FClearRequested := False ;
     with ToolServices.GetMainMenu do begin
        TargetItem := FindMenuItem( FILE_REOPEN_ID ) ;

        if TargetItem = nil then
           raise Exception.Create('Cannot find ' + FILE_REOPEN_ID + ' menu item') ;

        try
           ParentItem := TargetItem.GetParent ;
           Index := TargetItem.GetIndex ;
           {$IFDEF DEBUG}
           ShowMessage('Parent caption = ' + ParentItem.GetCaption) ;
           ShowMessage('Parent name = ' + ParentItem.GetName) ;
           ShowMessage('Caption = ' + TargetItem.GetCaption) ;
           ShowMessage('Index = ' + IntToStr(index)) ;
           {$ENDIF}
           Inc(Index) ;   // insert after
           if ParentItem = nil then
              raise Exception.Create('No menu parent for ' + FILE_REOPEN_ID ) ;
           try
              MenuOptions := [mfVisible] ;
              if mfEnabled in ParentItem.GetFlags then
                 MenuOptions := MenuOptions + [mfEnabled] ;
              FMenuItem := ParentItem.InsertItem(Index,
                                          'Clear &History',
                                          'FileClearClosedFiles',
                                          'This is a new menu item',
                                          0,
                                          0,
                                          0,
                                          MenuOptions,
                                          MenuClick) ;
           finally
              ParentItem.Free ;
           end ;
        finally
           TargetItem.Free ;
        end ;
     end ;
end ;

destructor TGLClearHistoryExpert.Destroy;
begin
     if FClearRequested then ClearHistory ;
     inherited Destroy ;
end;

procedure TGLClearHistoryExpert.MenuClick(Sender : TIMenuItemIntf) ;
begin
     if mfChecked in Sender.GetFlags then
        Sender.SetFlags( [mfChecked], [] )
     else
        Sender.SetFlags( [mfChecked], [mfChecked] ) ;
     FClearRequested := ( mfChecked in Sender.GetFlags ) ;
     if FClearRequested then
        MessageDlg( 'History will be cleared when you shut down the Delphi IDE', mtInformation, [mbOK], 0) ;
end ;

procedure TGLClearHistoryExpert.ClearHistory ;
var
   t : TRegIniFile ;
   slKeys : TStringList ;

                    procedure Process( sSection : string ) ;
                    var
                       x : integer ;
                    begin
                       t.ReadSection( sSection, slKeys ) ;
                       for x := 0 to slKeys.Count - 1 do
                          t.DeleteKey( sSection, slKeys[ x ] ) ;
                    end ;

begin
{$IFDEF VER120}
     t := TRegIniFile.Create('Software\Borland\Delphi\4.0') ;
{$ELSE}
   {$IFDEF VER130}
     t := TRegIniFile.Create('Software\Borland\Delphi\5.0') ;
   {$ELSE}
     t := TRegIniFile.Create('Software\Borland\Delphi\3.0') ;
   {$ENDIF}
{$ENDIF}
     slKeys := TStringList.Create ;
     try
        t.OpenKey( '', FALSE ) ;
        Process( 'Closed Files' ) ;
        Process( 'Closed Projects' ) ;
     finally
        t.CloseKey ;
        t.Free ;
        slKeys.Free ;
     end ;
end ;


procedure TGLClearHistoryExpert.Execute;
begin
     //
end;

function TGLClearHistoryExpert.GetAuthor: string;
begin
     Result := 'Greg Lief' ;
end;

function TGLClearHistoryExpert.GetComment: string;
begin
     Result := 'Greg Lief''s Clear History expert' ;
end;

// icons only apply to project or form experts
function TGLClearHistoryExpert.GetGlyph: HICON;
begin
     Result := 0 ;
end;

// global unique identifier for this expert
function TGLClearHistoryExpert.GetIDString: string;
begin
     Result := 'GLClearHistory.Expert' ;
end;

// menu label for Delphi's help menu.  In a real
// expert, you should load the string from a resource.
// Note that the stock D4 Help menu reserves the
// following trigger letters: A, C, I, N, P, R, S, T
function TGLClearHistoryExpert.GetMenuText: string;
begin
     Result := '' ;
end;

function TGLClearHistoryExpert.GetName: string;
begin
     Result := 'Greg Lief''s Clear History Expert' ;
end;

// name of page in object repository (only applies to form/project experts)
function TGLClearHistoryExpert.GetPage: string;
begin
     Result := '' ;
end;

// state of menu item (options include esEnabled and esChecked)
function TGLClearHistoryExpert.GetState: TExpertState;
begin
     Result := [esEnabled] ;
end;

// type of expert (esProject, esForm, esStandard, or esAddIn)
function TGLClearHistoryExpert.GetStyle: TExpertStyle;
begin
     Result := esAddIn ;
end;

procedure Register ;
begin
     RegisterLibraryExpert(TGLClearHistoryExpert.Create) ;
end ;

end.
